/** @file   wire.cpp
 * @brief   Implementation of Wire - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#include "wire.h"

namespace eng2d {

///
/// Static members, constants and datatypes
/// =======================================




///
/// Constructors, destructor and operators
/// ======================================
/** Constructor
 */
Wire::Wire( const Vec2D& begPos, const Vec2D& endPos, int nodecount ) :
  Interactive(),
  m_begPos( begPos ),
  m_endPos( endPos )
{
  this->createNodes( begPos, endPos, nodecount );
}



/** Destructor
 */
Wire::~Wire()
{
  for ( int i=0; i < m_nodes.size(); i++ ) {
    if ( m_nodes.at(i) != 0 ) {
      delete m_nodes.at(i);
    }
  }
  m_nodes.clear();
}




///
/// Public methods
/// ==============

/** Updates this wire
 */
void Wire::update()
{
}



/** Draws this wire
 */
void Wire::redraw( BITMAP* pB, const Vec2D& offset ) const
{
  if ( m_nodes.size() == 0 ) {
    Vec2D begP( m_begPos - offset );
    Vec2D endP( m_endPos - offset );
    line ( pB, begP.intX(),begP.intY(), endP.intX(),endP.intY(),
           m_color.allC() );
    return;
  }
  
  // Draw segmented line
  Vec2D segBeg( m_begPos - offset );
  Vec2D segEnd( m_nodes.at(0)->pos + m_nodes.at(0)->offset - offset );
  int c = m_color.allC();
  
  for ( int i=0; i <= m_nodes.size(); i++ ) {
    line ( pB, segBeg.intX(), segBeg.intY(), segEnd.intX(), segEnd.intY(), c );
    if ( i >= m_nodes.size() ) {
      break;
    }
    
    segBeg = m_nodes.at(i)->pos;
    segBeg += m_nodes.at(i)->offset;
    segBeg -= offset;
    
    if ( i+1 >= m_nodes.size() ) {
      segEnd = m_endPos;
      segEnd -= offset;
    } else {
      segEnd = m_nodes.at(i+1)->pos;
      segEnd += m_nodes.at(i+1)->offset;
      segEnd -= offset;
    }
  }
  return;
}



/** Sets the color.
 */
void Wire::color( const Color& rC )
{
  m_color = rC;
}



///
/// Getter methods
/// ==============

/** Returns the color.
 */
Color Wire::color() const
{
  return m_color;
}



/** Returns the nodecount
 */
int Wire::nodecount() const
{
  return m_nodes.size();
}




///
/// Private or Protected methods
/// ============================

/** Constructs given number of nodes between the two coordinates
 */
void Wire::createNodes( const Vec2D& rBeg, const Vec2D& rEnd, int nodes )
{
  float segs = nodes + 1; // number of line segments.
  
  Vec2D diffVec = rEnd - rBeg;
  float segLength = diffVec.length();
  segLength /= segs;
  
  diffVec.norm();
  diffVec *= segLength;
  
  Vec2D tmpPos = rBeg + diffVec;
  for ( int i=0; i < nodes; i++ ) {
    WireNode* pN = new WireNode;
    pN->pos = tmpPos;
    pN->offset = Vec2D(0,0);
    m_nodes.push_back( pN );
    
    tmpPos += diffVec;
  }
}


} // end of namespace
